//------------------------------------------------------------------------------
// File: svr_heroselect.cs
// This file contains methods to access and manipulate the current hero 
// selection
// Author: Matthew Rudge
// Copyright: Sandlot Games, 2007
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Called when the client selects a hero
//! \param %client Client that issued selection
//! \param %iHero Hero selected ($WW2HERO_*)
////////////////////////////////////////////////////////////////////////////////
function serverCmdOnHeroSelect(%client, %iHero)
{
   // Hero should be valid
   if(%iHero < 0) {
      return;
   }
   if(%iHero >= $WW2HERO_COUNT) {
      return;
   }
   
   // Select hero
   %loadedMission = $MissionLoaded;
   $MissionLoaded = true;
   slgSelectHero(%client, %iHero);
   $MissionLoaded = %loadedMission;
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the current hero selection
//! \param %iHero Hero selected ($WW2HERO_*)
////////////////////////////////////////////////////////////////////////////////
function slgSelectHero(%client, %iHero)
{
   // Reset current benefits, if any
   
   // Set current hero
   $SLGHERO_CURRENT = %iHero;

   // Set new benefits
   %heroDatablock = $HeroObject[$SLGHERO_CURRENT];
   %hero = CreateSaveObject(%heroDatablock, "HeroCharacter", $OST_PLAYER);
   %hero.setInternalName("HeroCharacter");
   
   %heroDatablock = %hero.getDataBlock();
   %hero.health = %heroDatablock.healthMax;
   %heroCount = slgGetSpawnCount($Point::Hero);
   
   // if there are no hero spawn points, spawn the hero at this position
   if (%heroCount == 0)
   {
      %hero.setPosition(-379.5, -377.5);
   }
   // otherwise, randomly pick a hero spawn point to spawn at from
   // the list of hero spawn points that have been created
   else
   {
      %heroIndex = getRandom(0, %heroCount - 1);
      %heroPt = slgGetSpawnPt($Point::Hero, %heroIndex).position;
      %hero.setPosition(getWord(%heroPt, 0), getWord(%heroPt, 1));
   }

   // Notify the client
   commandToClient(%client, 'HeroChanged', $SLGHERO_CURRENT);
}

////////////////////////////////////////////////////////////////////////////////
//! Determines if a particular hero is currently selected
//! \param %iHero Hero selection being queried ($WW2HERO_*)
//! \retval bool True if the hero is selected, false otherwise
////////////////////////////////////////////////////////////////////////////////
function slgIsHeroSelected(%iHero)
{
   return ($SLGHERO_CURRENT == %iChar);
}

////////////////////////////////////////////////////////////////////////////////
//! Moves the hero to his/her spawn point. Used to help the hero in case he/she
//! becomes 'stuck' on something
////////////////////////////////////////////////////////////////////////////////
function slgMoveHeroToSpawnPoint()
{
   %cmpChar = slgQueryInterface(HeroCharacter, $CID_CHARACTER);
   %cmpAI   = slgQueryInterface(HeroCharacter, $CID_AI);
   if(!isObject(%cmpChar) || !isObject(%cmpAI)) {
      return;
   }
   
   // Hero can not be repairing or building at the time
   if(%cmpChar.inState($CharacterState::Build) 
   || %cmpChar.inState($CharacterState::Repair)) {
      return;
   }
   
   // Hero can not be placing dynamite
   if(%cmpAI.toolActionsDisabled()) {
      return;
   }
   
   // Hero can not be healing
   if(HeroCharacter.isHealing || HeroCharacter.health <= 0) {
      return;
   }
   
   // Hero can not be in combat
   if(IsUnderAttack(HeroCharacter.getId())) {
      return;
   }
   if(%cmpAI.getCurrAction() $= "attack") {
      return;
   }
   
   %count = slgGetSpawnCount($Point::Hero);
   if(%count <= 0) {
      return;
   }
   
   // Just get the first hero point
   %spawnPt = slgGetSpawnPt($Point::Hero, 0);
   
   // Move hero character to hero point
   %spawnPos = %spawnPt.getPosition();
   HeroCharacter.setPosition(getWord(%spawnPos, 0), getWord(%spawnPos, 1), getWord(%spawnPos, 2));
   $serverCamera.moveToPosition(%spawnPos, true);
   
   // Stop all actions from occurring
   %cmpAI.popAllActions();
   
   // Stop the hero from moving
   StopMoving(%cmpAI);
   HeroCharacter.clearDestination();
   HeroCharacter.playThread(0, "root");
   RestoreCharacterOutfit(%cmpAI);
   
   // Hero must stop if doing a quick job
   if(%cmpChar.isCollecting()) {
      %cmpChar.stopCollecting();
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Command from client to move the hero to his/her spawn point
////////////////////////////////////////////////////////////////////////////////
function serverCmdMoveHeroToSpawnPoint()
{
   slgMoveHeroToSpawnPoint();
}

// End svr_heroselect.cs